#include "pch.h"

#include "MPlayer.h"

#include "MPlayerKernel.h"
#include "resource.h"

#define IDT_TIMER 1

static void ReportError(const wchar_t *msg) {
  MessageBox(nullptr, msg, L"ERROR", MB_OK | MB_ICONERROR);
}

static void ButtonSetIcon(HWND hBtn, HICON hIcon) {
  SendMessage(hBtn, BM_SETIMAGE, IMAGE_ICON, reinterpret_cast<LPARAM>(hIcon));
}

static void SliderInit(HWND hSlider, int min, int max, int lineSize) {
  SendMessage(hSlider, TBM_SETRANGEMIN, FALSE, min);
  SendMessage(hSlider, TBM_SETRANGEMAX, FALSE, max);
  SendMessage(hSlider, TBM_SETLINESIZE, TRUE, lineSize);
}

static void SliderSetPos(HWND hSlider, int pos) {
  SendMessage(hSlider, TBM_SETPOS, TRUE, pos);
}

static int SliderGetPos(HWND hSlider) {
  return static_cast<int>(SendMessage(hSlider, TBM_GETPOS, 0, 0));
}

static constexpr COMDLG_FILTERSPEC AudioFileFilter[] = {
    {L"MPEG Audio Layer 3", L"*.mp3"},
    {L"Free Lossless Audio Codec", L"*.flac"},
    {L"Wave", L"*.wav"},
    {L"Advanced Audio Coding", L"*.aac"},
    {L"OGG", L"*.ogg"},
    {L"All", L"*.*"},
};

struct IconSet {
  HICON file = nullptr;
  HICON play = nullptr;
  HICON pause = nullptr;
  HICON stop = nullptr;
  HICON volume = nullptr;
  HICON mute = nullptr;

  void Load(HINSTANCE hInst) {
    file = LoadIcon(hInst, MAKEINTRESOURCE(IDI_FILE));
    play = LoadIcon(hInst, MAKEINTRESOURCE(IDI_PLAY));
    pause = LoadIcon(hInst, MAKEINTRESOURCE(IDI_PAUSE));
    stop = LoadIcon(hInst, MAKEINTRESOURCE(IDI_STOP));
    volume = LoadIcon(hInst, MAKEINTRESOURCE(IDI_VOLUME));
    mute = LoadIcon(hInst, MAKEINTRESOURCE(IDI_MUTE));
  }
};

struct ControlSet {
  HWND filePath = nullptr;
  HWND progress = nullptr;
  HWND progressText = nullptr;
  HWND volume = nullptr;
  HWND openFile = nullptr;
  HWND mute = nullptr;
  HWND play = nullptr;
  HWND pause = nullptr;
  HWND stop = nullptr;

  void Load(HWND hDlg, const IconSet &icons) {
    filePath = GetDlgItem(hDlg, IDC_FILEPATH);
    progress = GetDlgItem(hDlg, IDC_PROGRESS);
    progressText = GetDlgItem(hDlg, IDC_PROGRESS_TEXT);
    volume = GetDlgItem(hDlg, IDC_VOLUME);
    openFile = GetDlgItem(hDlg, IDC_OPENFILE);
    mute = GetDlgItem(hDlg, IDC_MUTE);
    play = GetDlgItem(hDlg, IDC_PLAY);
    pause = GetDlgItem(hDlg, IDC_PAUSE);
    stop = GetDlgItem(hDlg, IDC_STOP);

    ButtonSetIcon(openFile, icons.file);
    ButtonSetIcon(play, icons.play);
    ButtonSetIcon(pause, icons.pause);
    ButtonSetIcon(stop, icons.stop);
  }
};

struct MPlayer::Impl {
public:
  Impl(HINSTANCE hInst) {
    mPlayerKernel = std::make_unique<MPlayerKernel>();
    mPlayerKernel->SetOnOpen([this]() { OnSourceOpen(); });
    mPlayerKernel->SetOnFinish([this]() { OnPlayFinish(); });
    mPlayerKernel->SetOnFailed([this]() { OnPlayFailed(); });

    LPARAM param = reinterpret_cast<LPARAM>(this);
    mhDialog = CreateDialogParam(hInst, MAKEINTRESOURCE(IDD_MPLAYER), nullptr, &Impl::DlgProc, param);
    InitDialog(hInst);
  }

  ~Impl() {
    if (mhDialog) {
      DestroyWindow(mhDialog);
    }
  }

  int Run(int nShowCmd) {
    if (!mhDialog) {
      ReportError(L"Failed to create MPlayer");
      return -1;
    }
    ShowWindow(mhDialog, nShowCmd);
    UpdateWindow(mhDialog);
    return MessageLoop(mhDialog);
  }

private:
  std::unique_ptr<MPlayerKernel> mPlayerKernel;

  HWND mhDialog = nullptr;

  IconSet mIcons;
  ControlSet mCtrls;

  bool mIsMute = false;
  int mVolume = 100;

  void InitDialog(HINSTANCE hInst) {
    mIcons.Load(hInst);
    mCtrls.Load(mhDialog, mIcons);

    SliderInit(mCtrls.volume, 0, 100, 10);
    SliderInit(mCtrls.progress, 0, 0, 5);

    SetMute(false);
    SetVolume(100);
  }

  void SetMute(bool isMute) {
    mIsMute = isMute;
    if (isMute) {
      SliderSetPos(mCtrls.volume, 0);
      ButtonSetIcon(mCtrls.mute, mIcons.mute);
    } else {
      SliderSetPos(mCtrls.volume, mVolume);
      ButtonSetIcon(mCtrls.mute, isMute ? mIcons.mute : mIcons.volume);
    }
    mPlayerKernel->SetMute(isMute);
  }

  void SetVolume(int volume) {
    mVolume = std::clamp(volume, 0, 100);
    SetMute(false);

    mPlayerKernel->SetVolume(mVolume);
  }

  void InitProgress() {
    int pos = static_cast<int>(mPlayerKernel->GetPosition().count());
    int length = static_cast<int>(mPlayerKernel->GetTotalLength().count());
    SliderInit(mCtrls.progress, 0, length, 5);
    UpdateProgressPos();
  }

  void UpdateProgressPos() {
    int pos = static_cast<int>(mPlayerKernel->GetPosition().count());
    int length = static_cast<int>(mPlayerKernel->GetTotalLength().count());
    SliderSetPos(mCtrls.progress, pos);

    wchar_t buffer[256] = {};
    wsprintf(buffer, L"%d:%02d / %d:%02d", pos / 60, pos % 60, length / 60, length % 60);
    SetWindowText(mCtrls.progressText, buffer);
  }

  void SetProgressPos(int pos) {
    mPlayerKernel->SetPosition(std::chrono::seconds(pos));
    UpdateProgressPos();
  }

  void OnVolumeScroll(int volume) {
    SetVolume(volume);
  }

  void OnProgressScroll(int pos) {
    SetProgressPos(pos);
  }

  void OnButtonMute() {
    SetMute(!mIsMute);
  }

  void OnSourceOpen() {
    InitProgress();
  }

  void OnPlayFinish() {
    OnStop();
  }

  void OnPlayFailed() {
    ReportError(L"Error occured during play");
    OnStop();
  }

  void OnButtonOpenFile() {
    winrt::com_ptr<IFileDialog> dialog = winrt::create_instance<IFileDialog>(CLSID_FileOpenDialog);
    dialog->SetFileTypes(std::size(AudioFileFilter), AudioFileFilter);

    if (SUCCEEDED(dialog->Show(nullptr))) {
      winrt::com_ptr<IShellItem> result;
      dialog->GetResult(result.put());

      PWSTR filePath = nullptr;
      result->GetDisplayName(SIGDN_FILESYSPATH, &filePath);
      mPlayerKernel->SetSource(filePath);

      SetWindowText(mCtrls.filePath, filePath);
      CoTaskMemFree(filePath);
    }
  }

  void OnPlay() {
    mPlayerKernel->Play();
    SetTimer(mhDialog, IDT_TIMER, 1000, nullptr);
  }

  void OnPause() {
    KillTimer(mhDialog, IDT_TIMER);
    mPlayerKernel->Pause();
  }

  void OnStop() {
    KillTimer(mhDialog, IDT_TIMER);
    mPlayerKernel->Stop();
    SetProgressPos(0);
  }

  void OnTimer() {
    UpdateProgressPos();
  }

  INT_PTR HandleMessage(UINT uMsg, WPARAM wParam, LPARAM lParam) {
    switch (uMsg) {
      case WM_TIMER: {
        if (wParam == IDT_TIMER) {
          OnTimer();
        }
        break;
      }
      case WM_HSCROLL: {
        HWND scroll = reinterpret_cast<HWND>(lParam);
        if (scroll == mCtrls.volume) {
          OnVolumeScroll(SliderGetPos(scroll));
        } else if (scroll == mCtrls.progress) {
          OnProgressScroll(SliderGetPos(scroll));
        }
        break;
      }
      case WM_COMMAND:
        switch (wParam) {
          case IDC_MUTE:
            OnButtonMute();
            break;

          case IDC_OPENFILE:
            OnButtonOpenFile();
            break;

          case IDC_PLAY:
            OnPlay();
            break;

          case IDC_PAUSE:
            OnPause();
            break;

          case IDC_STOP:
            OnStop();
            break;
        }
        break;

      case WM_CLOSE:
        PostQuitMessage(0);
        break;
    }
    return FALSE;
  }

  static int MessageLoop(HWND hDlg) {
    MSG msg = {};
    while (GetMessage(&msg, nullptr, 0, 0) != 0) {
      if (IsDialogMessage(hDlg, &msg)) {
        continue;
      }
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
    return 0;
  }

  static INT_PTR CALLBACK DlgProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
    if (uMsg == WM_INITDIALOG) {
      SetWindowLongPtr(hDlg, GWLP_USERDATA, static_cast<LONG_PTR>(lParam));
    }
    Impl *self = reinterpret_cast<Impl *>(GetWindowLongPtr(hDlg, GWLP_USERDATA));
    return self->HandleMessage(uMsg, wParam, lParam);
  }
};

MPlayer::MPlayer(HINSTANCE hInst) {
  mImpl = std::make_unique<Impl>(hInst);
}

MPlayer::~MPlayer() {}

int MPlayer::Run(int nShowCmd) {
  return mImpl->Run(nShowCmd);
}